﻿#include  "StdAfx.h"

#include  "CommandProcessor.hpp"
#include  "Command.hpp"
#include  "ProcessUI.hpp"
#include  "Decompressor.hpp"
#include  "Compressor.hpp"
#include  "NewArchiveUI.hpp"
#include  <szArchiveUpdater.hpp>
#include  <szRuntimeException.hpp>
#include  <szPath.hpp>
#include  <szString.hpp>
#include  <buffers.hpp>

#include  <boost/algorithm/string.hpp>
#include  <algorithm>
#include  <cassert>

SZ_AN_BEG

void SetProcessTitle(ProcessUI *ui, index_t cur, index_t count, const szstring &processName, const szstring &itemName)
{
  sbuf<szchar, MAX_PATH> buf;
  _stprintf_s(buf, buf.size(), SZL("[%d/%d] %s %s"), cur, count, processName.c_str(), itemName.c_str());
  ui->SetProcess((const szchar *)buf);
}

bool GetToggleState(const szstring &arg)
{
  // "on" と一致すれば true、それ以外なら false を返す
  const szstring larg(boost::algorithm::to_lower_copy(arg));
  return 0 == larg.compare(SZL("on"));
}

SZ_AN_END

using namespace szpp;

CommandProcessor::CommandProcessor(const szstring &cmdline, ProcessUI *pUI)
  : cmdline(cmdline), pUI(pUI), hThread(0), state(pUI)
{
  // スレッドに自分のポインタを渡して起動

  hThread = (HANDLE)_beginthreadex(NULL, 0, ThreadProc, this, CREATE_SUSPENDED, NULL);
  if (0 == hThread)
    BOOST_THROW_EXCEPTION(RuntimeException(SZT("Cannot create background thread")));

  ResumeThread(hThread);
}

CommandProcessor::~CommandProcessor()
{
  if (0 != hThread)
    CloseHandle(hThread);
}

// パスを標準化するメソッド。
// ・出力ディレクトリまたは現在のカレントディレクトリ（ディレクトリ遷移コマンドとかを実装して、コマンドプロセッサが管理するようにする予定）からの相対パスを絶対パスに変換
// ・ショートパスをロングパスに変換
//
// ちなみに、入力ファイルは原則としてフルパスが最初から得られているので、入力ディレクトリを指定するメリットはあまりない。
// 入力ファイルパスコマンドは削除してよいかも。
szstring CommandProcessor::Normalize(const szstring &path)
{
  szstring longPath = ConvertToLongPath(path);

  if (PATH_TYPE_RELATIVE != GetPathType(longPath))
    return longPath;

  szstring basePath = state.GetOutputPathRoot();
  if (basePath.empty())
  {
    sbuf<szchar, MAX_PATH> buf;
    if (0 == GetCurrentDirectory(buf.size(), buf))
      BOOST_THROW_EXCEPTION(RuntimeException(SZT("Cannot convert relative path to absolute path")));
    basePath.assign((const szchar *)buf);
  }
  return Normalize(Combine(basePath, longPath));
}

unsigned int WINAPI CommandProcessor::ThreadProc(void *lpParameter)
{
  CommandProcessor *pThis = reinterpret_cast<CommandProcessor *>(lpParameter);

  try
  {
    pThis->pUI->NotifyThread(pThis->hThread);
    pThis->pUI->SetEvents(&pThis->state.GetQuitEvent(), &pThis->state.GetDoneEvent());

    // コマンド解釈
    std::deque<Command> commands = ParseCommands(pThis->cmdline);

    // TODO: 前処理（圧縮の場合は圧縮ファイル数とか総データサイズを算出）、進行状況の基礎となる
    int processCount = 0;
    for (std::deque<Command>::size_type i = 0; i < commands.size(); ++i)
    {
      // 解凍なら解凍するアーカイブ数をくわえる
      if (DecompressCommand == commands[i].GetType())
        processCount += static_cast<int>(commands[i].GetArgs().size());
      // 圧縮ならアーカイブ数（一つ）をくわえる
      else if (CompressCommand == commands[i].GetType() || CompressAsCommand == commands[i].GetType())
        ++processCount;
    }

    // TODO: コマンド実行
    int currentProcess = 0;
    for (std::deque<Command>::size_type i = 0; i < commands.size(); ++i)
    {
      try
      {
        Command curCmd = commands[i];
        switch (curCmd.GetType())
        {
        case DecompressCommand:
          {
            for (std::vector<szstring>::size_type j = 0; j < curCmd.GetArgs().size(); ++j)
            {
              const szstring & curElem = pThis->Normalize(curCmd.GetArgs()[j]);

              // 進行状況の更新
              SetProcessTitle(pThis->pUI, ++currentProcess, processCount, SZT("Decompressing"), ExtractFileName(curElem));

              pThis->Decompress(curElem);
             
              if (pThis->Quitting())
                break;
            }
          }
          break;
        case CompressCommand:
        case CompressAsCommand:
          {
            bool compressAs = (CompressAsCommand == curCmd.GetType());
            szstring archivePath;
            ArchiveUpdaterOptions options;

            if (compressAs)
            {
              assert(1 < curCmd.GetArgs().size());

              // デフォルトの圧縮レベル
              options.Level = pThis->state.GetDefaultLevel();

              // 第一引数がアーカイブ名
              archivePath.assign(curCmd.GetArgs()[0]);
            }
            else
            {
              assert(1 <= curCmd.GetArgs().size());

              // 初期ディレクトリを設定（末尾にバックスラッシュを付けたディレクトリパス）
              archivePath = Combine(pThis->state.GetOutputPathRoot(), SZL(""));

              // アーカイブ名を問い合わせ
              NewArchiveUI naUI(pThis->pUI);
              if (!naUI.Show(&archivePath, &options))
              {
                // キャンセルされたので抜ける。おそらくこれ以降のコマンドも取りやめたいのがユーザーの意図だろうと考え、終了イベントもセット。
                pThis->state.GetQuitEvent().Set();
                BOOST_THROW_EXCEPTION(RuntimeException(SZT("Operation canceled by user")));
              }
            }

            std::vector<szstring> files(curCmd.GetArgs().begin() + (compressAs ? 1 : 0), curCmd.GetArgs().end());

            // 進行状況の更新
            SetProcessTitle(pThis->pUI, ++currentProcess, processCount, SZT("Compressing"), ExtractFileName(archivePath));

            pThis->Compress(pThis->Normalize(archivePath), files, &options);
           
            if (pThis->Quitting())
              break;
          }
          break;
        case InputPathCommand:
          assert(1 == curCmd.GetArgs().size());
          // 引数が相対パスなら無視
          if (GetPathType(curCmd.GetArgs()[0]) != PATH_TYPE_RELATIVE)
            pThis->state.SetInputPathRoot(curCmd.GetArgs()[0]);
          break;
        case OutputPathCommand:
          assert(1 == curCmd.GetArgs().size());
          // 引数が相対パスなら無視
          if (GetPathType(curCmd.GetArgs()[0]) != PATH_TYPE_RELATIVE)
            pThis->state.SetOutputPathRoot(curCmd.GetArgs()[0]);
          break;
        case LevelCommand:
          assert(1 == curCmd.GetArgs().size());
          if (curCmd.GetArgs()[0].size() == 1)
          {
            szchar c = curCmd.GetArgs()[0].at(0);
            if (c == SZL('1') || c == SZL('3') || c == SZL('5') || c == SZL('7') || c == SZL('9'))
              pThis->state.SetDefaultLevel(curCmd.GetArgs()[0]);
          }
          break;
        case VerboseCommand:
          assert(1 == curCmd.GetArgs().size());
          pThis->state.SetVerbose(GetToggleState(curCmd.GetArgs()[0]));
          break;
        }
      }
      // エラーが起きても次のコマンドの処理は続行できるように例外を捕捉。
      // TODO: RuntimeException をすべて捕捉するのが妥当かどうかは要検討。もしかすると致命度とかスキップ可能とか、例外にそういう分類が必要かも。
      catch (RuntimeException &runtimeException)
      {
        pThis->state.AddLog(runtimeException);
      }

      if (pThis->Quitting())
        break;
    }
  }
  // 基本的にここで補足される例外は無視も対処もできないので、メッセージを表示して終了。
  catch (RuntimeException &runtimeException)
  {
    pThis->state.AddLog(runtimeException);
  }
  catch (...)
  {
    pThis->state.AddLog(SZT("Error: Unknown error occurred"));
  }

  pThis->state.GetDoneEvent().Set();

  return TRUE;
}

bool CommandProcessor::Quitting() const
{
  return state.GetQuitEvent().Wait(0);
}

void CommandProcessor::Decompress(const szstring &archivePath)
{
  Decompressor decompressor(archivePath, &state, pUI);

  decompressor.Open();

  decompressor.Decompress();
}

void CommandProcessor::Compress(const szstring &archivePath, const std::vector<szstring> &files, ArchiveUpdaterOptions *options)
{
  Compressor compressor(files, archivePath, options, &state, pUI);

  compressor.Prepare();

  compressor.Compress();
}
